#!/bin/bash

set -e # Exit on any error

# Configuration
ANSIBLE_DIR="$HOME/dotfiles/orangepi/ansible"
VAULT_PASSWORD_FILE="$HOME/.ansible_vault_pass"

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Logging functions
log_info() {
  echo -e "${BLUE}[INFO]${NC} $1"
}

log_success() {
  echo -e "${GREEN}[SUCCESS]${NC} $1"
}

log_warning() {
  echo -e "${YELLOW}[WARNING]${NC} $1"
}

log_error() {
  echo -e "${RED}[ERROR]${NC} $1"
}

# Check if ansible is available
check_ansible() {
  if ! command -v ansible-playbook &> /dev/null; then
    log_error "ansible-playbook not found. Please install Ansible."
    exit 1
  fi
}

# Check vault password file
check_vault_password() {
  if [[ ! -f "$VAULT_PASSWORD_FILE" ]]; then
    log_warning "Vault password file not found at $VAULT_PASSWORD_FILE"
    log_info "You will be prompted for the vault password"
    VAULT_ARGS="--ask-vault-pass"
  else
    log_info "Using vault password file: $VAULT_PASSWORD_FILE"
    VAULT_ARGS="--vault-password-file $VAULT_PASSWORD_FILE"
  fi
}

# Show available services and tags
show_services() {
  echo "Available services and their tags:"
  echo ""
  echo "Service          Tags"
  echo "---------------  ------------------------"
  echo "homeassistant    homeassistant, smart-home"
  echo "duplicati        duplicati, backup"
  echo "homepage         homepage, dashboard"
  echo "media            media, jellyfin, sonarr, radarr, bazarr"
  echo "storage          storage, minio, syncthing"
  echo "immich-app       immich, photos"
  echo "vaultwarden      vaultwarden, passwords"
  echo "wg-easy          wg-easy, vpn, wireguard"
  echo "pihole           pihole, dns"
  echo "paperless        paperless, documents"
  echo ""
  echo "Tag groups:"
  echo "  backup         duplicati"
  echo "  media          jellyfin, sonarr, radarr, bazarr stack"
  echo "  vpn            wg-easy, wireguard"
  echo "  smart-home     homeassistant"
  echo "  photos         immich"
  echo "  passwords      vaultwarden"
  echo "  dns            pihole"
  echo "  documents      paperless"
}

# Show usage
show_usage() {
  echo "Usage: $0 [OPTIONS] [TAGS...]"
  echo ""
  echo "Deploy homelab services using Ansible with granular control"
  echo ""
  echo "Options:"
  echo "  -h, --help       Show this help message"
  echo "  -l, --list       List available services and tags"
  echo "  -s, --secrets    Only update secrets (.env files)"
  echo "  -d, --dry-run    Show what would be deployed without executing"
  echo "  -v, --verbose    Verbose output"
  echo ""
  echo "Examples:"
  echo "  $0                           # Deploy all services"
  echo "  $0 immich duplicati          # Deploy only Immich and Duplicati"
  echo "  $0 media vpn                 # Deploy media stack and VPN"
  echo "  $0 --secrets                 # Only update environment files"
  echo "  $0 --dry-run backup photos   # Show what backup and photos deployment would do"
  echo ""
  show_services
}

# Parse command line arguments
parse_args() {
  TAGS=()
  SECRETS_ONLY=false
  DRY_RUN=false
  VERBOSE=false
  
  while [[ $# -gt 0 ]]; do
    case $1 in
      -h|--help)
        show_usage
        exit 0
        ;;
      -l|--list)
        show_services
        exit 0
        ;;
      -s|--secrets)
        SECRETS_ONLY=true
        shift
        ;;
      -d|--dry-run)
        DRY_RUN=true
        shift
        ;;
      -v|--verbose)
        VERBOSE=true
        shift
        ;;
      *)
        TAGS+=("$1")
        shift
        ;;
    esac
  done
}

# Build ansible command
build_ansible_command() {
  local cmd="ansible-playbook"
  
  # Add inventory
  cmd="$cmd -i $ANSIBLE_DIR/inventory/local.yml"
  
  # Add playbook
  cmd="$cmd $ANSIBLE_DIR/playbook.yml"
  
  # Add vault arguments
  cmd="$cmd $VAULT_ARGS"
  
  # Add tags if specified
  if [[ ${#TAGS[@]} -gt 0 ]]; then
    local tag_string=$(IFS=,; echo "${TAGS[*]}")
    cmd="$cmd --tags $tag_string"
  fi
  
  # Add verbosity
  if [[ "$VERBOSE" == true ]]; then
    cmd="$cmd -v"
  fi
  
  # Add dry run
  if [[ "$DRY_RUN" == true ]]; then
    cmd="$cmd --check --diff"
  fi
  
  echo "$cmd"
}

# Main deployment function
main() {
  log_info "🚀 Starting Ansible-powered homelab deployment..."
  
  # Parse arguments
  parse_args "$@"
  
  # Check prerequisites
  check_ansible
  check_vault_password
  
  # Change to ansible directory
  cd "$ANSIBLE_DIR"
  
  # Build and execute ansible command
  local ansible_cmd=$(build_ansible_command)
  
  # Show what we're targeting
  if [[ ${#TAGS[@]} -gt 0 ]]; then
    local tag_string=$(IFS=,; echo "${TAGS[*]}")
    log_info "Targeting tags: $tag_string"
  else
    log_info "No tags specified, deploying all services"
  fi
  
  if [[ "$DRY_RUN" == true ]]; then
    log_info "DRY RUN MODE - No changes will be made"
  fi
  
  log_info "Executing: $ansible_cmd"
  echo ""
  
  # Execute the command
  if eval "$ansible_cmd"; then
    echo ""
    if [[ "$DRY_RUN" == true ]]; then
      log_success "Dry run completed successfully! 🏁"
    else
      log_success "Deployment completed successfully! 🎉"
      echo ""
      log_info "Next steps:"
      echo "  • Check service status: sudo systemctl status <service>-docker.service"
      echo "  • View logs: journalctl -u <service>-docker.service -f"
      echo "  • Access services via your configured domains"
    fi
  else
    log_error "Deployment failed! Check the output above for details."
    exit 1
  fi
}

# Run main function
main "$@"