require "digest"
require "rack"

# This class encapsulates a unit of work done for a particular tenant, connected to that tenant's database.
# ActiveRecord makes it _very_ hard to do in a simple manner and clever stuff is required, but it is knowable.
#
# What this class provides is a "misuse" of the database "roles" of ActiveRecord to have a role per tenant.
# If all the tenants are predefined, it can be done roughly so:
#
#     ActiveRecord::Base.legacy_connection_handling = false if ActiveRecord::Base.respond_to?(:legacy_connection_handling)
#     $databases.each_pair do |n, db_path|
#       config_hash = {
#         "adapter" => 'sqlite3',
#         "database" => db_path,
#         "pool" => 4
#       }
#       ActiveRecord::Base.connection_handler.establish_connection(config_hash, role: "database_#{n}")
#     end
#
#     def named_databases_as_roles_using_connected_to(n, from_database_paths)
#       ActiveRecord::Base.connected_to(role: "database_#{n}") do
#         query_and_compare!(n)
#       end
#     end
#
# So what we do is this:
#
# * We want one connection pool per tenant (per database, thus)
# * We want to grab a connection from that pool and make sure our queries use that connection
# * Once we are done with our unit of work we want to return the connection to the pool
#
# This also uses a stack of Fibers because `connected_to` in ActiveRecord _wants_ to have a block, but for us
# "leaving" the context of a unit of work can happen in a Rack body close() call.
class Shardine
  class Middleware
    def initialize(app, &database_config_lookup)
      @app = app
      @lookup = database_config_lookup
    end

    def call(env)
      connection_config = @lookup.call(env)

      # If lookup returns nil, skip database switching entirely
      if connection_config.nil?
        return @app.call(env)
      end

      switcher = Shardine.new(connection_config_hash: connection_config)
      did_enter = switcher.enter!
      status, headers, body = @app.call(env)
      body_with_close = Rack::BodyProxy.new(body) { switcher.leave! }
      [ status, headers, body_with_close ]
    rescue
      switcher.leave! if did_enter
      raise
    end
  end

  CONNECTION_MANAGEMENT_MUTEX = Mutex.new

  def initialize(connection_config_hash:)
    if ActiveRecord::Base.respond_to?(:legacy_connection_handling) && ActiveRecord::Base.legacy_connection_handling
      raise ArgumentError, "ActiveRecord::Base.legacy_connection_handling is enabled (set to `true`) and we can't use roles that way."
    end

    @config = connection_config_hash.to_h.with_indifferent_access
    @role_name = "shardine_#{@config.fetch(:database)}"
  end

  def with(&blk)
    create_pool_if_none!
    ActiveRecord::Base.connected_to(role: @role_name, &blk)
  end

  def enter!
    @fiber = Fiber.new do
      create_pool_if_none!
      ActiveRecord::Base.connected_to(role: @role_name) do
        Fiber.yield
      end
    end
    @fiber.resume
    true
  end

  def leave!
    to_resume, @fiber = @fiber, nil
    to_resume&.resume
    true
  end

  def create_pool_if_none!
    # Create a connection pool for that tenant if it doesn't exist
    CONNECTION_MANAGEMENT_MUTEX.synchronize do
      if ActiveRecord::Base.connection_handler.connection_pool_list(@role_name).none?
        ActiveRecord::Base.connection_handler.establish_connection(@config, role: @role_name)
      end
    end
  end
end

# # Use it like so:
# use Shardine::Middleware do |env|
#   site_name = env["SERVER_NAME"]
#   {adapter: "sqlite3", database: "sites/#{site_name}.sqlite3"}
# end
