class CustomersController < ApplicationController
  def index
    @customers = Customer.includes(:transactions)

    if params[:search].present?
      search_term = "%#{params[:search]}%"
      @customers = @customers.where("name LIKE ? OR email LIKE ?", search_term, search_term)
    end

    @customers = @customers.all
  end

  def show
    @customer = Customer.find(params[:id])
  end

  def new
    @customer = Customer.new
  end

  def create
    @customer = Customer.new(customer_params)

    if @customer.save
      redirect_to @customer, notice: "Customer created successfully"
    else
      render :new, status: :unprocessable_entity
    end
  end

  def edit
    @customer = Customer.find(params[:id])
  end

  def update
    @customer = Customer.find(params[:id])

    if @customer.update(customer_params)
      redirect_to @customer, notice: "Customer updated successfully"
    else
      render :edit, status: :unprocessable_entity
    end
  end

  def destroy
    @customer = Customer.find(params[:id])
    @customer.destroy
    redirect_to customers_path, notice: "Customer deleted successfully"
  end

  def record_visit
    @customer = Customer.find(params[:id])

    if @customer.visits_left <= 0
      redirect_to @customer, alert: "No visits left on this card"
      return
    end

    @customer.use_visit("Visit recorded via web interface")
    redirect_to @customer, notice: "Visit recorded successfully"
  end

  def purchase_card
    @customer = Customer.find(params[:id])
    visits = params[:visits].to_i

    if visits <= 0
      redirect_to @customer, alert: "Invalid number of visits"
      return
    end

    # Add expiration date (1 year from now)
    @customer.expires_at = 1.year.from_now.to_date
    @customer.save!

    @customer.purchase_visits(visits, "#{visits} visits purchased via web interface")
    redirect_to @customer, notice: "#{visits} visits added successfully"
  end

  private

  def customer_params
    params.require(:customer).permit(:name, :email, :visits_left, :expires_at)
  end
end
