import { Controller } from "@hotwired/stimulus"

// Connects to data-controller="modal"
export default class extends Controller {
  static targets = ["content"]

  connect() {
    // Bind escape key
    this.boundHandleEscape = this.handleEscape.bind(this)
    document.addEventListener("keydown", this.boundHandleEscape)
  }

  disconnect() {
    document.removeEventListener("keydown", this.boundHandleEscape)
    // Restore body scroll
    document.body.style.overflow = "auto"
  }

  show() {
    this.element.classList.remove("hidden")
    // Prevent body scroll
    document.body.style.overflow = "hidden"

    // Animate in
    requestAnimationFrame(() => {
      const container = this.element.querySelector(".modal-container")
      if (container) {
        container.classList.remove("translate-y-full")
      }
    })
  }

  hide() {
    const container = this.element.querySelector(".modal-container")
    if (container) {
      container.classList.add("translate-y-full")
    }

    // Wait for animation then hide
    setTimeout(() => {
      this.element.classList.add("hidden")
      document.body.style.overflow = "auto"

      // Clear content
      if (this.hasContentTarget) {
        this.contentTarget.innerHTML = ""
      }
    }, 300)
  }

  // Hide when clicking backdrop
  clickBackdrop(event) {
    if (event.target === this.element) {
      this.hide()
    }
  }

  // Hide on escape key
  handleEscape(event) {
    if (event.key === "Escape" && !this.element.classList.contains("hidden")) {
      this.hide()
    }
  }

  // Load content via Turbo Frame
  loadContent(event) {
    event.preventDefault()
    const url = event.currentTarget.href

    if (this.hasContentTarget) {
      // Create a temporary frame to fetch content
      const frame = document.createElement("turbo-frame")
      frame.id = "temp-frame"
      frame.src = url
      frame.style.display = "none"

      frame.addEventListener("turbo:frame-load", () => {
        // Move content from temp frame to modal
        this.contentTarget.innerHTML = frame.innerHTML
        frame.remove()
        this.show()
      })

      document.body.appendChild(frame)
    }
  }
}