# Simple tenant database seeder - no main database dependency
# This seeds individual tenant databases with users, customers, and transactions

def seed_tenant_database(tenant_subdomain)
  puts "\n🏢 Seeding #{tenant_subdomain} tenant database..."

  # Create Shardine instance for this tenant
  tenant_config = {
    adapter: "sqlite3",
    database: Rails.root.join("db", "tenant_#{tenant_subdomain}_#{Rails.env}.sqlite3")
  }

  shardine = Shardine.new(connection_config_hash: tenant_config)

  shardine.with do
    # Load tenant schema (includes users, sessions, customers, transactions)
    load(Rails.root.join("db", "tenant_schema.rb"))

    # Create admin user for this tenant
    admin_user = User.find_or_create_by!(email_address: "admin@#{tenant_subdomain}.com") do |u|
      u.password = "password123"
    end
    puts "  ✓ Admin user: #{admin_user.email_address}"

    # Create sample customers
    customers_data = [
      { name: "John Doe", email: "john@example.com", visits_left: 8, expires_at: 6.months.from_now },
      { name: "Jane Smith", email: "jane@example.com", visits_left: 3, expires_at: 3.months.from_now },
      { name: "Bob Wilson", email: "bob@example.com", visits_left: 15, expires_at: 1.year.from_now },
      { name: "Alice Brown", email: "alice@example.com", visits_left: 0, expires_at: 1.month.ago }
    ]

    customers_data.each do |customer_data|
      customer = Customer.find_or_create_by!(email: customer_data[:email]) do |c|
        c.name = customer_data[:name]
        c.visits_left = customer_data[:visits_left]
        c.expires_at = customer_data[:expires_at]
      end

      # Create realistic transaction history
      if customer.visits_left > 0
        # Purchase transaction
        purchase_amount = [ 10, 5, 20 ].sample
        customer.transactions.find_or_create_by!(transaction_type: :purchase, amount: purchase_amount) do |t|
          t.notes = "#{purchase_amount}-visit card purchase"
        end

        # Some visits used
        visits_used = purchase_amount - customer.visits_left
        visits_used.times do |i|
          customer.transactions.find_or_create_by!(
            transaction_type: :visit,
            amount: 1,
            created_at: (visits_used - i).days.ago
          ) do |t|
            t.notes = "Visit #{i + 1}"
          end
        end
      end

      puts "  ✓ #{customer.name} (#{customer.visits_left} visits left)"
    end
  end
end

# Seed the three test tenants
tenant_subdomains = [ "widgets", "acme", "testcorp" ]

puts "🌱 Seeding tenant databases..."
puts "="*50

tenant_subdomains.each do |subdomain|
  seed_tenant_database(subdomain)
end

puts "\n🎉 Seeding complete!"
puts "\nTest URLs:"
tenant_subdomains.each do |subdomain|
  puts "  http://#{subdomain}.lvh.me:3000"
  puts "    Admin login: admin@#{subdomain}.com / password123"
end
puts "\n127.0.0.1:3000 will default to widgets tenant"
